<?php
/**
 * Main functions library
 * 
 * Provides a bunch of useful functions
 * That file is included in both public / admin UIs
 * 
 * @package linea21\utils\lib
 * @author $Author$ - linea21 <info@linea21.com>
 * @version $Id$ 
 * @access public
 * @license http://opensource.org/licenses/gpl-3.0.html
 */


/**
 * implode_with_keys()
 * implode avec clefs associées renvoyées
 * sous forme de chaîne de caractères
 * @param string $glue
 * @param string $array
 * @return string
 */
function implode_with_keys($glue, $array) {
	$output = array();
	foreach( $array as $key => $item )
		$output[] = $key . "=" . $item;

	return implode($glue, $output);
}

/**
 * getHttpParameters()
 * Renvoie les paramètres HTTP
 * sous forme de chaîne de caractères
 * @return string
 */
function getHttpParameters($prefix = '?') {

	return $prefix. (string) implode_with_keys('&', $_REQUEST);
}

/**
 * is_module()
 * Test if the current page matches the given module and action (optionnal)
 * examples :  (1) is_module('system'), (2) is_module('system/edit_files')
 * interpreted as : (1) rub=system, (2) rub=system and todo=edit_files
 * @param string $str
 * @return boolean
 */
// @todo simo
function is_module($str) {

	$a = explode('/', $str);

	_debug('is_module() values : ' .join(', ', $a). ' '. count($a));

	// handle simple case :  is_module('system')
	if(!isset($_REQUEST['rub']) || strpos($a[0], $_REQUEST['rub']) === false) return false;

	// handle strict case : is_module('system$') $ - return false is $todo is set
	if(strpos($a[0], '$') !== false && isset($_REQUEST['todo'])) return false;

	// handle case is_module('system/list')
	if(isset($a[1]) && !empty($a[1])) {
		if(!isset($_REQUEST['todo']) || $_REQUEST['todo'] != $a[1]) return false;
	}

	return true;
}

/**
 * Remove DOM Nodes
 * @param string $html
 * @param string $tag
 * @return string
 */
function removeDomNodes($html, $tag)
{
	if(empty($html)) return $html;
	
    $html = str_get_html($html);
   
    foreach($html->find($tag) as $element) {
            $scripts = $element->outertext = '';
    }
    return $html;
}

/**
 * sanitize_string()
 * Convert html special chars  
 * @param string $str
 * @return string
 */
function sanitize_string($str) {
    $str = htmlspecialchars($str, ENT_COMPAT, CHARSET);
    return $str;
}

/**
 * strip_input()
 * Remove PHP and HTML code
 * @param string $str
 * @param string $exceptions
 * @return string
 */
function strip_input($str, $exceptions = false) {

    if(CURRENT_APP != 'admin') {
        $str = removeDomNodes($str, '//script');
        $str = strip_tags($str, '<p><strong><em><u><blockquote><table><tbody><tr><td><th><thead><tfoot><a><ul><ol><li><img>');
    }

    if(defined('RICH_TEXT_EDITOR') && RICH_TEXT_EDITOR != (string) 0) {
        return $str;
    }
    else {
        if(is_string($exceptions))  {
            return strip_tags($str, $exceptions);
        }
        if($exceptions === true)  {
            return strip_tags($str, ALLOWABLE_TAGS);
        }
        return strip_tags($str);
    }

}

/**
 * IncludeLightRte()
 * Include nicEdit as light RTE
 * param : array $a
 * @return void
 */
function IncludeLightRte($a) {

// @todo Remove if nicEdit is not used anymore in v 1.9
// Don't forget to remove $a parameter in fucntion and function call
// 	echo "<script type=\"text/javascript\">
// 					var nicEditorLang = '".U_L."';
// 				</script>".PHP_EOL;
// 	echo '<script type="text/javascript" src="../lib/js/nicEdit/lang.js"></script>'.PHP_EOL;
// 	echo '<script type="text/javascript" src="../lib/js/nicEdit/nicEdit.min.js"></script>'.PHP_EOL;
// 	echo "<script type=\"text/javascript\">
// 					bkLib.onDomLoaded(function() {
// 					new nicEditor({iconsPath : '../lib/js/nicEdit/nicEditorIcons.gif', buttonList : ['bold','italic','underline','subscript','superscript', 'link','unlink','image','upload']}).panelInstance('".$a['id']."');
// 				});
// 				</script>".PHP_EOL;

	$js = '$("#post-body").ckeditor(function() {
					// Instance loaded callback.
				}, {
					customConfig: "l21_light_config.js",
					language: "'.U_L.'",
					width: $(\'#post-title\').width() + 8
				});
				$("#post-body").after("<div class=\"form-info\">'._t('workshop', 'image_insert').'</div>");
				';

	echo '<script type="text/javascript" src="'.override('../lib/js/ckeditor/ckeditor.js', THEME_PUBLIC_PATH).'"></script>'.PHP_EOL;
	echo '<script type="text/javascript" src="'.override('../lib/js/ckeditor/adapters/jquery.js', THEME_PUBLIC_PATH).'"></script>'.PHP_EOL;

	footerAddInlineJS($js);
	
}

function write_curver() {
	file_put_contents('../config/.curver', LINEA_VERSION);
}

function get_curver() {
	return file_get_contents('../config/.curver');
}


function sql_update($filepath, $sqlo) {
	
	$handle = fopen($filepath, "r");
	if ($handle) {
		while (($line = fgets($handle)) !== false) {
			$line = trim($line);
			// we remove empty and commented lines
			if (!empty($line) && substr($line, 0, 2) !='--' &&  substr($line, 0, 1) !='#' ) {
				// we execute the query ignoring errors
				$r = $sqlo->DBQuery($line, true);
				// echo $line.'<br><br>';
			}
			
		}
	
		fclose($handle);
	} else {
		die('error opening SQL file : ' . $filepath);
	}
}

/**
 * IncludeTooltipJs()
 * Include jQuery tooltip plugin
 * @param string $selector
 * @return void
 */
function IncludeTooltipJs($selector) {
	$str='';
	if(is_string($selector)) {
		$str .= '$("'.$selector.'").tooltip({ effect: "slide", opacity: 0.8});';
	} else {
		foreach($selector as $sel) {
			$str .= '$("'.$sel.'").tooltip({ effect: "slide", opacity: 0.8});'.PHP_EOL;
		}
	}
	echo '<script type="text/javascript" src="../lib/js/jquery.tools.min.js"></script>'.PHP_EOL;
	echo '<script type="text/javascript">
	$(document).ready(function() {
	'.$str.'
})
</script>'.PHP_EOL;
}

/**
 * extRegex()
 * Transform a list given in the form "jpg,png,gif"
 * into "(jpg)|(png)|(gif)" format
 * @return string
 */
function extRegex($list) {
	$a = explode(',', $list);
	$b = array();
	foreach($a as $el) {
		array_push($b, '('.$el.')');
	}
	return implode('|', $b);
}

/**
 * dragTableSettings()
 * Include jQuery DragTable plugin settings
 * @return void
 */
function dragTableSettings() {

	$rub = isset($_REQUEST['rub']) ? $_REQUEST['rub'] : '';

	echo '<script type="text/javascript">
					$(document).ready(function(){
					$("table.sortable thead tr").prepend("<td>&nbsp;<\/td>");
					$("table.sortable tfoot tr").prepend("<td>&nbsp;<\/td>");
					$("table.sortable tbody tr").prepend("<td class=\"dragHandle\">&nbsp;<\/td>");
					$("table.sortable").tableDnD({
					onDrop: function(table, row) {
					$.get("../admin/_ajax_sort.php?rub='.$rub.'&" + $.tableDnD.serialize(), function(data) {
					// replacing data.class by data["class"] for IE8 bug fix
					if(data["class"] == "succeed") {
					alertify.success(data.msg);
				} else {
					alertify.error(data.msg);
				}
				},
				"json")
				},
				dragHandle: "dragHandle"
				});
				});
				</script>'.PHP_EOL;
}

/**
 * IncludeTreeJs()
 * Include jQuery tree plugin
 * @return void
 */
function IncludeTreeJs() {
	
	if(! is_module('scale') && ! is_module('sdi') && ! is_module('level')) return true;
	
	$str = '<ul id="cMenu" class="contextMenu"><li class="edit"><a href="#edit"><i class="icon-pencil"><\/i> '. _t('divers','modify').'<\/a><\/li><li class="delete separator"><a href="#delete"><i class="icon-trash"><\/i>  '. _t('divers','delete').'<\/a><\/li><\/ul>';
	echo '<script type="text/javascript" src="../lib/js/jquery.contextMenu/jquery.contextMenu.js"></script>'.PHP_EOL;
	echo '<link type="text/css" rel="stylesheet" href="../lib/js/jquery.contextMenu/jquery.contextMenu-l21.css" media="screen" />'.PHP_EOL;
	echo '<script type="text/javascript" src="../lib/js/jquery.treeview/jquery.treeview.js"></script>'.PHP_EOL;
	echo '<link type="text/css" rel="stylesheet" href="../lib/js/jquery.treeview/jquery.treeview.css" media="screen" />'.PHP_EOL;
	echo '<script type="text/javascript">
					$(document).ready(function(){
					$("#tree").after(\''.$str.'\');
					$("#tree").treeview({
					animated: "fast",
					collapsed: false,
					unique: false
				});

				$("#tree ul li.contextual-menu").contextMenu({
				menu: "cMenu"
				}, function(action, el, pos) {
				if(action=="edit") {
				if(typeof $(el).children("a.ico_mod").attr("href") == "undefined") {
				// not used anymore bu we keep it in case of
					alertify.error("'._t('msg', 'action_not_allowed').'");
					return 0;
				} else {
					document.location.href = $(el).children("a.ico_mod").attr("href");
				}
				}
				if(action=="delete") {
					if(typeof $(el).children("a.ico_sup").attr("href") == "undefined") {
						// used for scales - which does not have delete action on root
						alertify.error("'._t('msg', 'action_not_allowed').'");
						return 0;
					} else {
						document.location.href = $(el).children("a.ico_sup").attr("href");
					}
				}

				});
				$("#tree li a.ico_mod, #tree li a.ico_sup").css("display", "none");

				});
				</script>'.PHP_EOL;
}

function includeCodeEditor() {

	// we exit function if not enabled
	if(!defined('EDITOR_HIGHLIGHT_CODE') || EDITOR_HIGHLIGHT_CODE == 0 )  return true;

	// we exit function if not the concerned module
	if(!is_module('system/edit_files')) return true;

	if(isset($_REQUEST['file'])) {
		$ext = pathinfo($_REQUEST['file'], PATHINFO_EXTENSION);
	} else {
		$ext = 'php';
	}

	$header = '';

	$header .=  '<link type="text/css" rel="stylesheet" href="'.override('../lib/js/CodeMirror/lib/codemirror.css', THEME_ADMIN_PATH).'" />'.PHP_EOL;
	$header .=  '<script type="text/javascript" src="'.override('../lib/js/CodeMirror/lib/codemirror.js', THEME_ADMIN_PATH).'"></script>'.PHP_EOL;
	$header .=  '<script type="text/javascript" src="'.override('../lib/js/codeMirror-fullscreen.js', THEME_ADMIN_PATH).'"></script>'.PHP_EOL;
	$header .=  '<script type="text/javascript" src="'.override('../lib/js/CodeMirror/addon/selection/active-line.js', THEME_ADMIN_PATH).'"></script>'.PHP_EOL;

	if($ext == 'css') {
		$header .=  '<script type="text/javascript" src="'.override('../lib/js/CodeMirror/mode/css/css.js', THEME_ADMIN_PATH).'"></script>'.PHP_EOL;

		$code = 'var editor = CodeMirror.fromTextArea(document.getElementById("file_content"), {
							styleActiveLine: true,
							viewportMargin: Infinity,
							lineNumbers: true,
							extraKeys: {
				        "F11": function(cm) {
				          setFullScreen(cm, !isFullScreen(cm));
				        },
				        "Esc": function(cm) {
				          if (isFullScreen(cm)) setFullScreen(cm, false);
				        }
				      }
						});';

		footerAddInlineJS($code);
	}

	if($ext == 'php') {
		$header .=  '<script type="text/javascript" src="'.override('../lib/js/CodeMirror/addon/edit/matchbrackets.js', THEME_ADMIN_PATH).'"></script>'.PHP_EOL;
		$header .=  '<script type="text/javascript" src="'.override('../lib/js/CodeMirror/mode/htmlmixed/htmlmixed.js', THEME_ADMIN_PATH).'"></script>'.PHP_EOL;
		$header .=  '<script type="text/javascript" src="'.override('../lib/js/CodeMirror/mode/xml/xml.js', THEME_ADMIN_PATH).'"></script>'.PHP_EOL;
		$header .=  '<script type="text/javascript" src="'.override('../lib/js/CodeMirror/mode/javascript/javascript.js', THEME_ADMIN_PATH).'"></script>'.PHP_EOL;
		$header .=  '<script type="text/javascript" src="'.override('../lib/js/CodeMirror/mode/css/css.js', THEME_ADMIN_PATH).'"></script>'.PHP_EOL;
		$header .=  '<script type="text/javascript" src="'.override('../lib/js/CodeMirror/mode/clike/clike.js', THEME_ADMIN_PATH).'"></script>'.PHP_EOL;
		$header .=  '<script type="text/javascript" src="'.override('../lib/js/CodeMirror/mode/php/php.js', THEME_ADMIN_PATH).'"></script>'.PHP_EOL;

		$code = 'var editor = CodeMirror.fromTextArea(document.getElementById("file_content"), {
							styleActiveLine: true,
			        lineNumbers: true,
			        matchBrackets: true,
			        mode: "application/x-httpd-php",
			        indentUnit: 4,
			        indentWithTabs: true,
			        enterMode: "keep",
			        viewportMargin: Infinity,
			        tabMode: "shift",
			        extraKeys: {
				        "F11": function(cm) {
				          setFullScreen(cm, !isFullScreen(cm));
				        },
				        "Esc": function(cm) {
				          if (isFullScreen(cm)) setFullScreen(cm, false);
				        }
				      }
			      });';
		
		footerAddInlineJS($code);
	}

	echo $header;

}

function IncludeRichTextEditor() {

	if(defined('RICH_TEXT_EDITOR') && RICH_TEXT_EDITOR != (string) 0 )  {
		if(strtolower(RICH_TEXT_EDITOR) == 'tinymce')
		{

			if(CURRENT_APP == 'admin') {
				if(isset($_REQUEST['rub']) && $_REQUEST['rub'] == 'newsletter' && file_exists(THEME_PUBLIC_PATH. 'css/newsletter.css')) {
					// Loads CSS files into the current document
					echo '<script type="text/javascript">
					newslettercsspath = "'. THEME_PUBLIC_PATH . 'css/newsletter.css";
					</script>';
				}
			}
			
			echo '<script type="text/javascript" src="'.override('../lib/js/tinymce/js/tinymce/tinymce.min.js', THEME_ADMIN_PATH).'"></script>'.PHP_EOL;
			echo '<script type="text/javascript" src="'.override('../lib/js/tinymce/js/tinymce/config.js', THEME_ADMIN_PATH).'"></script>'.PHP_EOL;
		}
		
		if(strtolower(RICH_TEXT_EDITOR) == 'cke')
		{
			echo '<script type="text/javascript" src="'.override('../lib/js/ckeditor/ckeditor.js', THEME_ADMIN_PATH).'"></script>'.PHP_EOL;
			echo '<script type="text/javascript" src="'.override('../lib/js/ckeditor/adapters/jquery.js', THEME_ADMIN_PATH).'"></script>'.PHP_EOL;

			$extraConfig = '';
			
			if(CURRENT_APP == 'admin') {
				if(isset($_REQUEST['rub']) && $_REQUEST['rub'] == 'newsletter' && file_exists(THEME_PUBLIC_PATH. 'css/newsletter.css')) {

					$extraConfig .= ',
						contentsCss: "'.THEME_PUBLIC_PATH. 'css/newsletter.css"';
				}
			}

			if(defined('CKFINDER_ENABLED') && CKFINDER_ENABLED == true) {
				echo '<script type="text/javascript" src="'.override('../lib/vendor/ckfinder/ckfinder.js', THEME_ADMIN_PATH).'"></script>'.PHP_EOL;
				$callback = 'CKFinder.setupCKEditor( this, "../lib/vendor/ckfinder/");';
			} else {
				$callback = '';
				$extraConfig .= ',
					filebrowserBrowseUrl: "../library/access.php",
					filebrowserImageBrowseUrl: "../library/access.php?type=images",
					filebrowserWindowWidth: "80%",
					filebrowserWindowHeight: "80%"';
			}

			$js = '$(".largetextfield").ckeditor(function() {
					'.$callback.'
				}, {
					customConfig: "l21_config.js"'.$extraConfig.',
					footnotesPrefix: "a",
					footnotesTitle: "'._t('common', 'footnotes'). '"
				});';
			
			footerAddInlineJS($js);
				
		}
	}

}


/**
 * IncludeColorboxJs()
 * Include Colorbox jQuery plugin
 * @param string £selector
 * @return void
 */
function IncludeColorboxJs($selector) {
	$str='';
	if(is_string($selector)) {
		$str .= '$("'.$selector.'").not(".dlfile").colorbox({transition:"fade"});'.PHP_EOL;
	} else {
		foreach($selector as $sel) {
			$str .= '$("'.$sel.'").not(".dlfile").colorbox({transition:"fade"});'.PHP_EOL;
		}
	}
	echo '<script type="text/javascript" src="../lib/js/colorbox/jquery.colorbox-min.js"></script>'.PHP_EOL;
	echo '<link type="text/css" rel="stylesheet" href="../lib/js/colorbox/colorbox.css" />'.PHP_EOL;
	echo '<script type="text/javascript">
				$(document).ready(function() {
				'.$str.'
			});
</script>'. PHP_EOL;

}

/**
 * includeLeafletHeader()
 * Include the leaflet header
 * if needed
 * @return void
 */
function includeLeafletHeader() {

	$GLOBALS['mapping_includes'] = array('directory-detail|', 'yellowpages|det');

	$rub = isset($_REQUEST['rub']) ? $_REQUEST['rub'] : '';
	$todo = isset($_REQUEST['todo']) ? $_REQUEST['todo'] : '';

	if(ActiveItemKey($rub)) $rub = ActiveItemKey($rub);
	$s = $rub . '|' . $todo;
	$a = $GLOBALS['mapping_includes'];

	if(in_array($s, $a)) {
		echo '<script src="http://cdn.leafletjs.com/leaflet-0.7.2/leaflet.js"></script>'.PHP_EOL;
		echo '<link rel="stylesheet" href="http://cdn.leafletjs.com/leaflet-0.7.2/leaflet.css" />'.PHP_EOL;
		echo '<style>
					.leaflet-popup-content-wrapper {
						-webkit-border-radius: 0 ! important;
						-moz-border-radius: 0 ! important;
						border-radius: 0 ! important;
					}
					</style>';
	}
}

/**
 * SureCreateDir()
 * Créer un dossier s'il n'existe pas.
 * @param string $pathname
 * @param integer $perms
 * @return integer $ver_num
 */
function SureCreateDir($pathname, $perms) {
	if(!file_exists($pathname)) {
		return mkdir ($pathname, $perms, true);
	} else {
		return true;
	}
}


/**
 * SureRemoveDir()
 * Supprime le contenu d'un dossier et le dossier lui-même si précisé.
 *
 * @return integer $ver_num
 */
function SureRemoveDir($dir, $DeleteMe) {
	if(!$dh = @opendir($dir)) return;
	while (($obj = readdir($dh))) {
		if($obj=='.' || $obj=='..') continue;
		if (!@unlink($dir.'/'.$obj)) SureRemoveDir($dir.'/'.$obj, true);
	}
	if ($DeleteMe){
		closedir($dh);
		@rmdir($dir);
	}
}

/**
 * num_phpversion()
 * Retourne un entier comme numéro de version PHP
 *
 * @return integer $ver_num
 */
function num_phpversion() {
	$ver = explode( '.', phpversion() );
	$ver_num = $ver[0] . $ver[1] . $ver[2];

	return $ver_num;
}

/** mb_ucfirst()
 * UTF-8 ucfirst function
 * @param string $string
 * @param string $encoding
 * @return string
 */
function mb_ucfirst($string, $encoding = CHARSET)
{
	if(function_exists('mb_strlen')) {
		$strlen = mb_strlen($string, $encoding);
		$firstChar = mb_substr($string, 0, 1, $encoding);
		$then = mb_substr($string, 1, $strlen - 1, $encoding);
		return mb_strtoupper($firstChar, $encoding) . $then;
	} else {
		_debug('mb_string module not loaded', 'warning');
		return ucfirst($string);
	}
}

/**
 * cutText()
 * Découpe un texte à une longeur donnée.
 *
 * @param string $content
 * @param integer $length
 * @param integer $abbr
 * @param string $end
 * @return
 */
function cutText($content, $length, $abbr = 0, $end = ' ...')
{
	
	$content_light =  Stringy\Stringy::create($content, CHARSET)->safeTruncate($length, $end);
	
	if (strlen($content) > $length && $abbr == 1) {
		$content_light = "<abbr title=\"" . $content . "\">" . $content_light . "</abbr>\n";
	}
	
	return $content_light;
}

/**
 * error_redirect()
 * Redirect to error page
 */
function error_redirect($file ='error.php') {

	header("Location: " . CURRENT_APP_URL . $file);
  exit();
}

/**
 * removeEmptyP()
 * Remove empty P tags
 *
 * @param string $text
 * @return array $body
 */
function removeEmptyP($str) {
	$str = preg_replace('#<p[^>]*>(\s|&nbsp;?)*</p>#', '', $str);

	return $str;
}

/**
 * formatNavTitle()
 * Formatage des titres ( interface admin )
 *
 * @param string $content
 * @return string $content
 */
function formatNavTitle($content)
{
	$content = formatText($content, '2HTML');
	$content = cutText($content, 70, 1);

	return $content;
}

/**
 * formatTextli()
 * Formatage des listes ( interface admin )
 *
 * @param string $content
 * @return string $content
 */
function formatTextli($content)
{
	$content = formatText($content, '2HTML');
	$content = cutText($content, 70, 1);
	return $content;
}

/**
 * formatTitleh2()
 * Formatage des titres h2 ( interface admin )
 *
 * @param string $content
 * @return string $content
 */
function formatTitleh2($content)
{
	$content = formatText($content, '2HTML');
	return $content;
}

/**
 * isRawText()
 * check if raw text or html text
 *
 * @param string $t
 * @return bool
 */
function isRawText($t) {
	if (strcmp($t, strip_tags($t)) == 0) {
		return true;
	} else {
		return false;
	}
}

/**
 * formatText()
 * Formatage de texte pour affichage
 *
 * @param  $content
 * @param string $format
 * @return string $content
 */
function formatText($content, $format = -1)
{
	$content = stripslashes(trim($content));
	switch ($format) {
		case '2HTML':
			if(RICH_TEXT_EDITOR === 0 || isRawText($content)) $content = nl2br($content);
			break;
		case '2FIELD':
			$content = htmlentities($content, ENT_QUOTES, 'utf-8');
			break;
		case '2ATT':
			$content = htmlentities($content, ENT_QUOTES, 'utf-8');
			break;
		case '2XML':
			$content = htmlspecialchars($content);
			break;
		case '2FILE':
			//$content = addslashes(trim($content));
			$content = htmlspecialchars($content, ENT_QUOTES, 'utf-8');
			break;
		default:
	}
	return $content;
}

function stripText($text) {

	$text = strtolower($text);

	// strip all non word chars
	$text = preg_replace('/\W/', ' ', $text);
	// replace all white space sections with a dash
	$text = preg_replace('/\ +/', '-', $text);
	// trim dashes
	$text = preg_replace('/\-$/', '', $text);
	$text = preg_replace('/^\-/', '', $text);

	return $text;
}

/**
 * toStringSqlDate()
 * Renvoie la date au format SQL
 *
 * @param string $format
 * @return string $s
 */
function toStringSqlDate($format = 'short')
{
	$date_format =   array(
			'dd-mm-yyyy' => array(
					'mysql' => array('short'=> '%d-%m-%Y', 'long'=>
							array('12' => '%d-%m-%Y  %r', '24' => '%d-%m-%Y  %T')),
					'pgsql' => array('short'=> 'DD-MM-YYYY', 'long'=>
							array('12' => 'DD-MM-YYYY HH12:MI:SS', '24' => 'DD-MM-YYYY HH24:MI:SS'))),
			'yyyy-mm-dd' => array(
					'mysql' => array('short'=> '%Y-%m-%d', 'long'=>
							array('12' => '%Y-%m-%d  %r', '24' => '%Y-%m-%d  %T')),
					'pgsql' => array('short'=> 'YYYY-MM-DD', 'long'=>
							array('12' => 'YYYY-MM-DD HH12:MI:SS', '24' => 'YYYY-MM-DD HH24:MI:SS'))),

	);

	if($format == 'long') $s = $date_format[DATE_FORMAT][SQL][$format][TIME_FORMAT];
	else $s = $date_format[DATE_FORMAT][SQL][$format];

	return $s;
}


/**
 * date_compare()
 * Compare 2 dates with a given operator.
 * @param  $date1
 * @param  $date2
 * @param  $op
 * @return boolean
 */
function date_compare($date1, $date2, $op) {

	$date1= strtotime(formatDate($date1, true));
	$date2= strtotime(formatDate($date2, true));

	switch($op) {
		case '>':
			if($date1 > $date2) return true;
			else return false;
			break;
		case '<':
			if($date1 < $date2) return true;
			else return false;
		case '>=':
			if($date1 >= $date2) return true;
			else return false;
			break;
		case '<=':
			if($date1 <= $date2) return true;
			else return false;
		case '==':
			if($date1 == $date2) return true;
			else return false;
		default:
			return false;
	}
}

/**
 * ln10filename()
 * Build a localized filename
 * according to the current language
 *
 * @param string $file
 * @return string
 */
function ln10filename($file)
{

	$tmp=@explode(".", $file);
	$total = count($tmp) - 1;
	$ext = $tmp[$total];
	unset($tmp[$total]);

	return @implode(".", $tmp). '.' . U_L. '.' .$ext;
}


/**
 * distInclude()
 * Include the required file
 * if no user file is found,
 * includes the dist/ version file.
 * Localized files have the priority
 *
 * @param string $file
 * @return void
 */
function distInclude($file, $default_dist)
{
	$l10n_file = ln10filename($file);
	$l10n_file_dist =  dirname($l10n_file).'/dist/'.basename($l10n_file);

	$file_dist_default =  $default_dist.'dist/'.basename($file);
	$l10n_file_dist_default = $default_dist.'dist/'.basename($l10n_file);

	if(file_exists($l10n_file)) {
		_debug('distInclude() file Inclusion : '.$l10n_file);
		include_once($l10n_file);

	} elseif(file_exists($l10n_file_dist))  {
		_debug('distInclude() file Inclusion : '.$l10n_file_dist);
		include_once($l10n_file_dist);
	}
	elseif(file_exists($file)) {
		_debug('distInclude() file Inclusion : '.$file);
		include_once($file);
	}
	elseif(file_exists($l10n_file_dist_default)) {
		_debug('distInclude() file Inclusion : '.$l10n_file_dist_default);
		include_once($l10n_file_dist_default);
	}
	elseif(file_exists($file_dist_default)) {
		_debug('distInclude() file Inclusion : '.$file_dist_default);
		include_once($file_dist_default);
	}
	else {
		_debug('distInclude() file Inclusion : '.dirname($file).'/dist/'.basename($file));
		include_once(dirname($file).'/dist/'.basename($file));
	}

}

/**
 * getOverridePluginsFiles()
 * return an array of paths
 * corresponding to the given filename
 * and matching the array of given plugins 
 *
 * @param array $a
 * @param string $filename
 * @return array
 */
function getOverridePluginsFiles($a, $filename) {

	$o = array();
	if(count($a) == 0) return $o;

	foreach ($a as &$v) {
		$path = '../plugins/'.$v.'/override/'.str_replace('../', '', $filename);

		if(is_file($path) && file_exists($path)) {
			array_push($o, $path);
		}
	}
	return $o;

}

/**
 * override()
 * check if override version exists for the whole app
 * then if a template file version exists or not
 * finally, if no user file is found in theme,
 * return the default module version file.
 *
 * @param string $file
 * @param string $path
 * @return string
 */
function override($file, $path = null, $default = true)
{
	// we first check if file is handled by a plugin
	$paths = getOverridePluginsFiles(enabledPlugins(), $file);

	if(count($paths) > 0) {
		if(count($paths) > 1 && is_file($file)) {
			_debug('[plugins override] possible conflicts : '.count($paths). ' occurences of the same file. Loaded from '. $paths[0], 'warning');
		}
		_debug('[plugins override] : '.$paths[0]);
		return $paths[0];
	}

	// if not, we check if file exists in OVERRIDE_FOLDER
	$filename = OVERRIDE_FOLDER.str_replace('../', '', $file);
	if(file_exists($filename)) {
		_debug('[general override] : '.$filename);
		return $filename;

	}

	// if not again we are checking into themes folders
	if(is_null($path)) {
		$path = THEME_PUBLIC_PATH;
	}

	$theme_file = $path.'override/'.str_replace('../', '', $file);

	if(file_exists($theme_file)) {
		_debug('[thematic override] : '.$theme_file);
		return $theme_file;

		// finally we include the default one if asked
	}
	if($default) {
		_debug('[no override] : '.$file);
		return $file;
	}

	return true;

}


/**
 * formatDate()
 * Renvoie la date aux formats yyyy-mm-dd ou dd-mm-yyyy suivant le cas de départ
 * Si $db == true renvoie toujours la date au format yyyy-mm-dd
 *
 * @param string $date
 * @param boolean $db
 * @return string $new_date
 */
function formatDate($date, $db = false)
{
	@list($part1, $part2, $part3) = explode('-', $date);
	if(strlen($part1) == 2) {
		$new_date = $part3 . '-' . $part2 . '-' . $part1;
	} else {
		$new_date = $part1 . '-' . $part2 . '-' . $part3;
	}
	if($db == true) {
		// always return yyyy-mm-dd format
		if(strlen($part1) == 2) {
			$new_date = $part3 . '-' . $part2 . '-' . $part1;
		} else {
			$new_date = $part1 . '-' . $part2 . '-' . $part3;
		}
	}

	return $new_date;
}


/**
 * date_rfc2822()
 * Format date to RFC 2822 date format
 * @param string $date
 * @return string (exemple : Thu, 21 Dec 2000 16:01:07 +0200)
 */
function date_rfc2822($date) {
	if(!isNullDate($date)) {
		$tmp_date = formatDate($date, true);
		@list($y, $m, $d) = explode('-', $tmp_date);
		return date("r", mktime(3, 0, 0, $m, $d, $y));
	} else {
		return false;
	}
}


function isNullDate($date)
{
	if(substr($date, 0, 10) == '0001-01-01' || substr($date, 0, 10) == '01-01-0001') {
		return true;
	} else return false;

}

/**
 * empty_nc()
 * retourne le contenu ou N.C
 *
 * @param string $content
 * @return string $content
 */
function empty_nc($content)
{
	$content = trim($content);
	if ($content=='' || isNullDate($content)) $content = _t('divers','nc');

	return $content;
}

/**
 * empty_none()
 * retourne le contenu ou 'aucun'
 *
 * @param string $content
 * @return string $content
 */
function empty_none($content)
{
	$content = trim($content);
	if (empty($content)) $content = _t('divers','none');

	return $content;
}

/**
 * empty_none()
 * retourne le contenu ou 0
 *
 * @param string $content
 * @return string $content
 */
function empty_numeric($content)
{
	$content = trim($content);
	if (empty($content)) $content = 0;

	return $content;
}

/**
 * checkdate_validity()
 * Vérifie la validité d'une date
 *
 * @param string $date
 * @param string $msg (optionnal)
 * @return boolean true or error message (string)
 */
function checkdate_validity($date, $msg= '')
{

	if(empty($date)) return  $msg ._t('date','do_provide');

	$date=formatDate($date, true);
	@list($year, $month, $day) = explode('-', $date);

	if(!preg_match('/^\d{4}-\d\d-\d\d$/', $date)) {
		return $msg . _t('date','not_valid');
	}

	if (!@checkdate($month , $day , $year)) return $msg . _t('date','date_do_not_exist');
	return true;
}

/**
 * sortItems()
 * Return an array with key and value
 * with ID as key and RANGE as value
 * @return array
 */
function sortItems($rub, $data) {

	$a = array();

	for($i=0; $i<count($data); $i++) {
		$v = $data[$i];
		$a[$v]= $i+1;
	}

	return $a;

}

/**
 * display_errors()
 * Affichage d'un message d'erreur utilisateur
 *
 * @param string $msg
 * @return void (echo)
 */
function display_errors($msg)
{
	$display_it = "<div class=\"error\">\n";
	$display_it .= $msg;
	$display_it .= "</div>\n";
	echo $display_it;
}

/**
 * system_error()
 * Affichage d'un message d'erreur système
 *
 * @param string $msg
 * @return void (echo)
 */
function system_error($msg = ERROR_SYSTEM)
{
	$display_it = "<div class=\"systemerror\">\n";
	$display_it .= $msg;
	$display_it .= "</div>\n";
	echo $display_it;
	exit;
}

/**
 * get_min_name()
 * obtenir le nom de la miniature d'un fichier
 *
 * @param string $path
 * @return string $min_path
 * @todo can be removed later / still used in class.upload.php
 */
function get_min_name($path)
{
	$short_path = dirname($path);
	$filename = basename($path);
	$min_path = $short_path . "/min_" . $filename;
	return $min_path;
}

/**
 * ExcedMaxSize()
 * Teste si une image dépasse ou non la taille autorisée (en pixels)
 *
 * @param string $path
 * @param integer $width_max
 * @return boolean
 */
function ExcedMaxSize($path, $width_max)
{
	list($width, $height, $type, $attr) = getimagesize($path);
	if ($width > $width_max || $height > $width_max) return true;
	else return false;
}

/**
 * cancel_button()
 * génére un bouton de retour
 *
 * @param  $back_uri
 * @return string
 */
function cancel_button($back_uri)
{
	return '<input name="annuler" type="button" value="' . _t('btn','annul') . '" class="button" id="annuler" onclick="window.location=\'' . $back_uri . '\';" />';
}

/**
 * cancel_button_close_parent()
 * génére un bouton de retour
 *
 * @param  $back_uri
 * @return string
 */
function cancel_button_close_parent()
{
	return '<input name="annuler" type="button" value="' . _t('btn','annul') . '" class="button" id="annuler" onclick="parent.$.fn.colorbox.close();" />';
}

/**
 * GetDisplayUserRight()
 * renvoie les droits d'un utilisateur
 *
 * @param string $indice
 * @param string $module
 * @return string
 */
function GetDisplayUserRight($indice, $module = -1)
{
	$indice = strtoupper($indice);
	if ($indice == 'U') return _t('user','norights');
	if ($indice == 'A') return _t('user','adminrights');
	if ($indice == 'O' && $module == 'dashboard') return _t('user','managerrights');
	if ($indice == 'O' && $module == 'project') return _t('user','managerrights');
	if ($indice == 'O' && $module == 'workshop') return _t('user','animatorrights');
	if ($indice == 'O' && ($module != 'workshop' && $module != 'dashboard')) return _t('user','redactorrights');
}

/**
 * get_linkin_page()
 *
 * @param array $table_link
 * @param integer $total
 * @param integer $debut
 * @param integer $pas
 * @return void
 **/
function get_linkin_page($table_link, $total, $debut, $pas = SELECT_LIMIT)
{
	$result = ceil($total / $pas);
	if ($result <= 1) return '&nbsp;';
	else {
		$link = '<div class="pagination">'.PHP_EOL;

		// Previous arrow pagination
		if(($debut - $pas) >= 0) {
		$array_pos = array ('debut' => $debut - $pas);
		$new_table_link = array_merge ($table_link, $array_pos);
		$link .= '<a href="'. HrefMaker($new_table_link) . '">«</a>'.PHP_EOL;
		} else {
			$link .= '« ';
		}
		$sep='';
		for($i = 0; $i < $result; $i++) {
			$current_pos = ($pas * $i);
			if ($debut == $current_pos) $link .= $sep."<span>" . ($i + 1) . "</span> \n";
			else {
				$array_pos = array ('debut' => $current_pos);
				$new_table_link = array_merge ($table_link, $array_pos);
				$link .= $sep.'<a href="'. HrefMaker($new_table_link) . '">' . ($i + 1) . '</a>'.PHP_EOL;
			}
			// $sep=' | ';
		}
		// Next arrow pagination 
		if(($debut + $pas) < $total) {
			$array_pos = array ('debut' => $debut + $pas);
			$new_table_link = array_merge ($table_link, $array_pos);
			$link .= '<a href="'. HrefMaker($new_table_link) . '">»</a>'.PHP_EOL;
		} else {
			$link .= ' »';
		}
		$link .= '</div>'.PHP_EOL;

		return $link;
	}
}

/**
 * linkin_page()
 * création d'un navigateur de pages numérotées
 *
 * @param string $string_uri
 * @param integer $total
 * @param integer $debut
 * @param integer $pas
 * @return string $link
 */
function linkin_page($string_uri, $total, $debut, $pas = SELECT_LIMIT)
{
	$result = ceil($total / $pas);
	if ($result <= 1) return '&nbsp;';
	else {
		if (strpos($string_uri, '?') === false) $string_uri .= '?';
		else $string_uri .= '&amp;';
		$link = '<div class="pagination">';

		// Previous arrow pagination
		if(($debut - $pas) >= 0) {
			$pos = $debut - $pas;
			$link .= '<a href="'. $string_uri . 'debut=' . $pos . '">«</a>'.PHP_EOL;
		} else {
			$link .= '« '.PHP_EOL;
		}

		for($i = 0; $i < $result; $i++) {
			$current_pos = ($pas * $i);
			if ($debut == $current_pos) $link .= "<span>" . ($i + 1) . "</span> \n";
			else $link .= "<a href=\"" . $string_uri . "debut=" . $current_pos . "\">" . ($i + 1) . "</a> \n";
		}

		// Next arrow pagination
		if(($debut + $pas) < $total) {
			$pos = $debut + $pas;
			$link .= '<a href="'. $string_uri . 'debut=' . $pos . '">»</a>'.PHP_EOL;
		} else {
			$link .= ' »'.PHP_EOL;
		}

		$link .= '</div>';
		return $link;
	}
}

/**
 * display_statut()
 * renvoie le statut en pleines lettres
 *
 * @param string $statut
 * @return string $result
 */
function display_statut($statut)
{
	switch ($statut) {
		case 'P':
			$result = _t('statut','public');
			break;
		case 'D':
			$result = _t('statut','draft');
			break;
		case 'E':
			$result = _t('statut','E');
			break;
		case 'AA':
			$result = _t('statut','AA');
			break;
		case 'PA':
			$result = _t('statut','PA');
			break;
		case 'C':
			$result = _t('statut','C');
			break;
		case 'U':
			$result = _t('statut','U');
			break;
		case 'O':
			$result = _t('statut','O');
			break;
		case 'A':
			$result = _t('statut','A');
			break;
		case 'W':
			$result = _t('statut','W');
			break;
		default:
			$result = _t('statut','public');
	}
	return mb_ucfirst($result);
}

/**
 * sys_number_format()
 * returns a given number formatted the 'system' way
 * for example a french float 25,68 would become 25.68
 * @param string $statut
 * @return string $result
 */
function sys_number_format($number) {
	if(LANGUAGE == "fr") {
		$number = str_replace(',', '.', $number);
		$number = str_replace(' ', '', $number);
	}
	return $number;
}

function fnumber_format($number, $dec, $forceprefix = true) {

	if(!is_numeric($number)) return $number;
	
	$number = str_replace(' ', '', $number);

	// we determine decimal numbers
	// if 'auto' option is passed
	if($dec == 'auto') {
		$number = (string) $number;
		if(!strpos($number, '.')) {
			$dec = 0;
		} else {
			$dec = strlen(substr(strrchr($number, '.'), 1));
		}
	}

	if(!is_numeric($number)) return '';

	if(LANGUAGE == "fr") {
		$number = number_format($number, $dec, ',', ' ');
	} else {
		$number = number_format($number, $dec, '.', ',');
	}

	// adding space to negative symbol - removed jquery.tablesorter does not sort if there is one space
	//$number = str_replace('-', '- ', $number);

	// adding '+' symbol if positive
	if(is_numeric($number[0]) && $forceprefix) $number = '+' . $number;

	return $number;

}

/**
 * linkin_content()
 * Cherche les liens et emails dans du contenu -> linkage
 *
 * @param string $content
 * @param string $option
 * @return string $content
 */
function linkin_content($content, $option = 'ALL')
{
	if(defined('AUTO_LINK') && AUTO_LINK == 1)
	{
		if(defined('RICH_TEXT_EDITOR') && RICH_TEXT_EDITOR == (string) 0) {
			if ($option == 'ALL' || $option == 'MAIL') {
				$content = eregi_replace("( |<br />)+([_a-z0-9-]+(\.[_a-z0-9-]+)*@[a-z0-9-]+(\.[a-z0-9-]+)+)", '\\1<a href="mailto:\\2">\\2</a>', $content);
			}
			if ($option == 'ALL' || $option == 'LINK') {
				$content = eregi_replace("(http|https|ftp|ftps)://([-a-z0-9#?/&=:,_;@%.{}]*)([a-z0-9=]{2,4})", '<a href="\\1://\\2\\3" class="out">\\1://\\2\\3</a>', $content);
			}
		}
	}

	return $content;
}

/**
 * QuickBoxNow()
 * Génération de la quickbox
 *
 * @param string $module
 * @param string $h1
 * @param string $liste
 * @param string $suffixclass
 * @return string $quickbox
 */
function QuickBoxNow($module, $h1, $liste, $suffixclass = '')
{
	$quickbox = "<div class=\"entete\">\n<div class=\"qb_ico\" id=\"qbico" . $suffixclass . "\" title=\"" . $module . "\"></div>\n";
	$quickbox .= "<div class=\"quickbox\" id=\"qbbg" . $suffixclass . "\">\n";
	$quickbox .= "<h1>" . $h1 . "</h1>\n";
	$quickbox .= "<ul>";
	$quickbox .= $liste;
	$quickbox .= "</ul>";
	$quickbox .= "</div>";
	$quickbox .= "</div>";

	return $quickbox;
}

/**
 * redirect_to()
 * redirect to module's default page
 * or given module
 *
 * @param string $rub
 * @return void
 */
function redirect_to($module = '@module_default')
{
	if($module == '@module_default') {
		$module = $_REQUEST['rub'];
	}
	
	return header("Location: index.php?rub=".$module);

}

/**
 * ReloadIndex()
 * Chargement de l'index après destruction de sessions
 *
 * @param string $item
 * @return void
 */
function ReloadIndex($item)
{
	switch ($item) {
		case 'public':
			return header("Location: ../public/index.php");
			break;
		case 'admin':
			return header("Location: ../admin/logout.php");
			break;
	}
}

/**
 * getmicrotime()
 * renvoie le temps en microsecondes
 *
 * @return float
 */
function getmicrotime()
{
	list($usec, $sec) = explode(" ", microtime());
	return ((float)$usec + (float)$sec);
}


/**
 * availablePlugins()
 * Return available plugins
 * @return array
 **/
function availablePlugins() {
	$a = array();
	if ($handle = opendir('../plugins/')) {
		while (false !== ($file = readdir($handle))) {
			if (substr($file, 0, 4) == 'l21_' ) {
				array_push($a, $file);
			}
		}
		closedir($handle);
	}

	return $a;
}

/**
 * enabledPlugins()
 * Return available plugins
 * @return array
 **/
function enabledPlugins() {
	$a = array();
	if ($handle = opendir('../plugins/')) {
		while (false !== ($file = readdir($handle))) {
			if (substr($file, 0, 4) == 'l21_' && file_exists('../plugins/'.$file.'/.active')) {
				array_push($a, $file);
			}
		}
		closedir($handle);
	}

	return $a;
}


/**
 * array_extract()
 * 
 * @param array $inputArray
 * @param string $key
 * @return array
 * @todo Not that clean : would be nice to remove the RecursiveIteratorIterator part
 */
function array_extract($inputArray, $needle, $recursive = false, $identifier = false) {
	
	$simpleArray = array();
	$flattenArray = array();
	
	foreach ($inputArray as $key => &$entry) {
		
		// echo "$key <br ><br>";
		// where only interested in numeric items
		// as those are the actual children
		if( !is_numeric( $key ) )
		{
			// otherwise continue
			continue;
		}
		// echo $entry[$needle]. " / ";
		$simpleArray[] = $entry[$needle];
		//echo '<br> <br>Array : '. implode(' / ', $simpleArray).'<br> <br>';
		if($recursive && isset($entry[$identifier])) {
			$item = array_extract($entry[$identifier], $needle, $recursive, $identifier);
			if ($item) {
				$simpleArray[] = $item;
			}
		}
	}
	// return $simpleArray;
	// $simpleArray return Array ( [0] => 6 [1] => Array ( [0] => 56 [1] => Array ( [0] => 57 ) ) [2] => 5 [3] => 4 [4] => 3 ) 
	// we flatten it to Array ( [0] => 6 [1] => 56 [2] => 57 [3] => 5 [4] => 4 [6] => 3 ) 
	$it = new RecursiveIteratorIterator(new RecursiveArrayIterator($simpleArray));
	foreach($it as $v) {
		array_push($flattenArray, $v);
	}
	return $flattenArray;
}


function getLevelChildren($data, $id) {

	$res = array();

	foreach ($data as $item) {

		if($item['level_parent'] == $id) {
			$children = getLevelChildren($data, $item['level_id']);
			if ($children) {
				$item['children'] = $children;
			}
			$res[] = $item;
		}
	}
	return $res;
}


/**
 * sql_dump2array()
 * @param $url
 * @param $a
 * @return array
 * @link http://fr2.php.net/manual/fr/function.mysql-query.php
 */
function sql_dump2array($url, $a = -1) {

	$handle = @fopen($url, "r");
	$query = "";
	if($a == -1) $a = array();

	while(!feof($handle)) {
		$sql_line = fgets($handle);
		if (trim($sql_line) != "" && strpos($sql_line, "--") === false && strpos($sql_line, "#") === false) {
			array_push($a, $sql_line);
		}
	}
	return $a;
}

/**
 * sql_status_filter()
 * @param $fieldname
 * @param $a array
 * @return string
 */
function sql_status_filter($fieldname, $a) {

	$str = '';
	$sep = '';

	foreach ($a as $value) {
		$str.= $sep. $fieldname . " = '". $value ."'";
		$sep = " OR ";
	}
	if(count($a) > 1) $str ='('. $str .')';

	return $str;
}

/**
 * getDynamicProgressbar()
 * Return a graphic progress bar generated by easy-pie-chart
 * @param $value integer
 * @param $size integer
 * @return string
 */
function getDynamicProgressbar($value, $size = 110) {


	// include easy-pie-chart js library
	footerAddJS('../lib/js/easy-pie-chart/dist/jquery.easypiechart.min.js');
	footerAddJS('../lib/js/waypoints/lib/jquery.waypoints.min.js');
	footerAddInlineJS("$('#progession-chart').waypoint( {
	handler: function(direction) {
		$(this.element).easyPieChart({
			size:'".$size."',
			lineWidth:'15',
			trackColor: '#ccc',
			barColor:'#456F7F',
			lineCap:'butt',
			onStep: function(from, to, percent) {
				$(this.el).find('.percent').text(Math.round(percent));
			}
		});
		this.destroy();
	}, 
	offset: 'bottom-in-view'  
});");
	
	$str   = 	'<span id="progession-chart" data-percent="'.$value.'">'.PHP_EOL;
	$str  .=	'<span class="percent"></span>'.PHP_EOL;
	$str  .=	'</span>'.PHP_EOL;

	return $str;
}

/**
 * geocode()
 * get Long/Latitude coordinates
 * @param $address string
 * @return object
 */
function geocode($address) {

	$o = new StdClass; // prevent PHP 'Warning: Creating default object from empty value'
	$o->{'status'} = 'geocoder_disabled';

	if(!defined('GEOCODER_ENABLED') || GEOCODER_ENABLED == 0)
		return $o->{'status'};

	$address=str_replace(" ","+",$address);
	if ($address) {
		$json = file_get_contents(GEOCODER_URL.'address='.$address.'&sensor=false');

		$o = json_decode($json);
	}

	return $o;
}

/**
 * placeholderReplace()
 * format '{$key}'
 * @param $array
 * @param $input
 * @return array
 */
function placeholderReplace($array, $input) {
	foreach ($input as $key => $value)
	{
		$array = preg_replace("/{{$key}}/i", $value, $array);
	}
	return $array;
}

/**
 * generateRandomString()
 * @param integer $length
 * @return string
 */
function generateRandomString($length = 10) {
	
	$characters = '0123456789abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ';
	$randomString = '';
	
	for ($i = 0; $i < $length; $i++) {
		$randomString .= $characters[rand(0, strlen($characters) - 1)];
	}
	
	return $randomString;
}

/**
 * GenerateXhtmlPage()
 *
 * @param $string
 * @param integer $time
 * @param unknown $redirect
 * @return
 **/
function GenerateXhtmlPage($string, $time = 2, $redirect = CURRENT_APP_URL)
{


	$page  = '<?xml version="1.0" encoding="' . CHARSET . '" xml:lang="' . U_L . '"?>' . PHP_EOL;
	$page .= '<!DOCTYPE html PUBLIC "-//W3C//DTD XHTML 1.1//EN" "http://www.w3.org/TR/xhtml11/DTD/xhtml11.dtd">' . PHP_EOL;
	$page .= '<html xmlns="http://www.w3.org/1999/xhtml" xml:lang="' . U_L . '">' . PHP_EOL;
	$page .= '<head>' . PHP_EOL;
	$page .= '<meta http-equiv="content-type" content="text/html;charset=' . CHARSET . '" />' . PHP_EOL;
	$page .= '<meta http-equiv="content-langage" content="' . CHARSET . '" />' . PHP_EOL;
	$page .= '<meta http-equiv="refresh" content="' . $time . ';url=' . $redirect . '">' . PHP_EOL;
	$page .= '<link rel="icon" type="image/gif" href="' .THEME_ADMIN_PATH. 'images/favicon.gif" />' . PHP_EOL;
	$page .= '<title>'.SITE_NAME.'</title>' . PHP_EOL;
	$page .= '</head>' . PHP_EOL;
	$page .= '<body>' . PHP_EOL;
	$page .= '<div class="info" style="margin-top:10em">'.$string.PHP_EOL;
	$page .= '<br />';
	$page .= sprintf( _t('divers','redirect_string'), $redirect, $time);
	$page .= '</div>'. PHP_EOL;
	$page .= '</body>' . PHP_EOL;
	$page .= '</html>' . PHP_EOL;

	return $page;
}

/**
 * _debug()
 * Display a debug message
 * @param string
 * @return void
 **/
function _debug($string, $messagetype = 'system')
{
	if(isset($GLOBALS['debugbar']))
	{
		global $debugbar;

		$debugbar["messages"]->addMessage($string, $messagetype);
		
	}
}

/**
 * headerAddCSS()
 * Add CSS into header
 * @param string
 * @param string
 * @return boolean
 **/
function headerAddCSS($path, $pos = 'default')
{
	if(!fopen($path, 'r')) {
		_debug('<b>Problem loading CSS file</b> : '.$path . ' (position : '.$pos.')', 'error');
		return false;
	}

	if(isset($GLOBALS['CSSheader']) && is_numeric(strpos($GLOBALS['CSSheader'], $path))) {
		_debug('<b>Warning : CSS file already loaded</b> :'.$path, 'warning');
		return false;
	}

	$str = '<link type="text/css" rel="stylesheet" href="'.$path.'" />'.PHP_EOL;

	if(isset($GLOBALS['CSSheader']))
	{
		if($pos == 'first') {
			$GLOBALS['CSSheader'] = $str . PHP_EOL . $GLOBALS['CSSheader'] ;
		} else {
			$GLOBALS['CSSheader'] .=  PHP_EOL . $str;
		}
	} else {
		$GLOBALS['CSSheader'] = $str;
		
	}
	_debug('Loading <b>CSS file</b> : '.$path . ' (position : '.$pos.')');

	return true;
}

/**
 * headerAddJS()
 * Add JS into header
 * @param string
 * @param string
 * @return boolean
 **/
function headerAddJS($path, $pos = 'default')
{
	if(!fopen($path, 'r')) {
		_debug('<b>Problem loading JS file</b> : '.$path . ' (position : '.$pos.')', 'error');
		return false;
	}

	if(isset($GLOBALS['JSheader']) && in_array($path , $GLOBALS['JSheader'])) {
		_debug('<b>Warning : JS file already loaded</b> :'.$path, 'warning');
		return false;
	}

	$str= '<script type="text/javascript" src="'.$path .'"></script>'.PHP_EOL;

	if(isset($GLOBALS['JSheader']))
	{
		if($pos == 'first') {
			$GLOBALS['JSheader'] = $str . $GLOBALS['JSheader'];
		}
		else {
			$GLOBALS['JSheader'] .= $str . PHP_EOL ;
		}
	} else {
		$GLOBALS['JSheader'] = $str;
	}
	_debug('Loading <b>JS file</b> : '.$path . ' (position : '.$pos.')');

	return true;
}

/**
 * footerAddInlineJS()
 * Add inline JS into footer
 * the resulting js is wrapped with <script> tags and jQuery .ready() function if $wrap == true
 * @param string
 * @param string
 * @return boolean
 **/
function footerAddInlineJS($str, $pos = 'default', $wrap = true)
{
if($wrap == true) {
	if(isset($GLOBALS['JSInlinefooter']))
	{
		if($pos == 'first') {
			$GLOBALS['JSInlinefooter'] = $str . PHP_EOL. $GLOBALS['JSInlinefooter'];
		}
		else {
			$GLOBALS['JSInlinefooter'] .= $str . PHP_EOL ;
		}
	} else {
		$GLOBALS['JSInlinefooter'] = $str. PHP_EOL;
	}
} else {
	if(isset($GLOBALS['JSInlinefooter_nowrap']))
	{
		if($pos == 'first') {
			$GLOBALS['JSInlinefooter_nowrap'] = $str . PHP_EOL. $GLOBALS['JSInlinefooter_nowrap'];
		}
		else {
			$GLOBALS['JSInlinefooter_nowrap'] .= $str . PHP_EOL ;
		}
	} else {
		$GLOBALS['JSInlinefooter_nowrap'] = $str. PHP_EOL;
	}
}
	_debug('Loading <b>inline JS</b> : '.cutText($str, 90) . ' (position : '.$pos.')');

	return true;
}

/**
 * footerAddJS()
 * Add JS into footer
 * @param string
 * @param string
 * @return boolean
 **/
function footerAddJS($path, $pos = 'default')
{
	if(!fopen($path, 'r')) {
		_debug('<b>Problem loading JS file</b> : '.$path . ' (position : '.$pos.')', 'error');
		return false;
	}

	if(isset($GLOBALS['JSfooter']) && is_numeric(strpos($GLOBALS['JSfooter'], $path))) {
		_debug('<b>Warning : JS file already loaded</b> :'.$path, 'warning');
		return false;
	}

	$str= '<script type="text/javascript" src="'.$path .'"></script>'.PHP_EOL;

	if(isset($GLOBALS['JSfooter']))
	{
		if($pos == 'first') {
			$GLOBALS['JSfooter'] = $str . $GLOBALS['JSheader'];
		}
		else {
			$GLOBALS['JSfooter'] .= $str . PHP_EOL ;
		}
	} else {
		$GLOBALS['JSfooter'] = $str . PHP_EOL ;
	}
	_debug('Loading <b>JS file</b> : '.$path . ' (position : '.$pos.')');

	return true;
}

function addDynamicCSS($url) {

	// use of document.createStyleSheet for IE8 compatibility
	$str = 'if (document.createStyleSheet)
					{
						document.createStyleSheet("'.$url. '");
					}
					else
					{
						$(\'<link rel="stylesheet" type="text/css" href="'.$url. '" />\').appendTo("head");
					}';

	footerAddInlineJS($str);

}

/**
 * AddDynamicHeader()
 * Display JS and CSS header
 * @return void
 **/
function AddDynamicHeader() {
	if(isset($GLOBALS['JSheader'])) echo $GLOBALS['JSheader'];
	if(isset($GLOBALS['CSSheader']))  echo $GLOBALS['CSSheader'];
	
	if(defined('MAP_ENGINE') && MAP_ENGINE == 'leaflet') includeLeafletHeader();
}

/**
 * AddDynamicFooter()
 * Display JS footer
 * @return void
 **/
function AddDynamicFooter() {
	if(isset($GLOBALS['JSfooter'])) echo $GLOBALS['JSfooter'];
	if(isset($GLOBALS['JSInlinefooter_nowrap'])) echo $GLOBALS['JSInlinefooter_nowrap'];
	if(isset($GLOBALS['JSInlinefooter'])) {
		echo '<script type="text/javascript">'.PHP_EOL;
		echo '	$(document).ready(function() {'.PHP_EOL;
		echo $GLOBALS['JSInlinefooter'];
		echo '  });'.PHP_EOL;
		echo '</script>'.PHP_EOL;
	}
}


function format_version_number($val) {
	$val = preg_replace("/(-rev|-r)([0-9])+/",'', $val); // remove revision number (ie "1.5-rev1815" will return "1.5")
	$val = preg_replace("/([^0-9])/",'', $val); // remove letter (if dev version)
	$val = str_replace('.', '', $val);
	
	if(strlen($val) == 2) $val = str_pad ($val ,3, '0', STR_PAD_RIGHT);
	
	return $val;
}

/**
 * check4newVersion()
 * Display a link to download new version if available
 * @return string
 **/
function check4newVersion() {
	
	$url = '?name='.rawurlencode(SITE_NAME).'&url='.rawurlencode(SITE_ROOT_URL).'&email='.rawurlencode(SITE_MAIL);
			
	try{
		if(!@$rss=simplexml_load_file(SITE_LINEA_URL.'/linea21-version-check.php'. $url)) {
			throw new Exception('Version : xml file was not found');
		}

		$current_version = format_version_number(LINEA_VERSION);
		$latest_version = format_version_number($rss->num);
		
 		// _debug('current ver : '.$current_version . ' latest ver : '.$latest_version);

		if((integer)$latest_version > (integer)$current_version) {
			$update = '<div id="version-check"><span>'._t('check_update','search').' : </span><a href="'.(string)$rss->link.'">'.sprintf(_t('check_update','dl'), $rss->num).'</a></div>';
		} else {
			$update = '<div id="version-check"><span>'._t('check_update','search').' : '._t('check_update','ok').'</span></div>';
		}
	}
	catch(Exception $e){
		$update = $e->getMessage();
	}

	return $update;
}

/**
 * loadThemeInfo()
 * Load theme info
 * @param string
 * @param string
 * @return string
 **/
function loadThemeInfo($type, $name) {
	$a = array();
	try{
		if(!@$flow=simplexml_load_file(SITE_PATH.'templates/'.$type.'/'.$name.'/theme.xml')){
			throw new Exception($name.' plugin : xml file was not found');
		}
		$a['name'] = $flow->themename;
		$a['description'] = $flow->description;
		$a['version'] = $flow->version;
		$a['date'] = $flow->date;
		$a['compatibility'] = $flow->compatibility;
		$a['author'] = $flow->author;
		$a['homepage'] = $flow->homepage;

		return $a;
	}
	catch(Exception $e){
		return $e->getMessage();
	}
}

/**
 Validate an email address.
 Provide email address (raw input)
 Returns true if the email address has the email
 address format and the domain exists.
 @link http://www.linuxjournal.com/article/9585
 */
function validEmail($email)
{
	$isValid = true;
	$atIndex = strrpos($email, "@");
	if (is_bool($atIndex) && !$atIndex)
	{
		$isValid = false;
	}
	else
	{
		$domain = substr($email, $atIndex+1);
		$local = substr($email, 0, $atIndex);
		$localLen = strlen($local);
		$domainLen = strlen($domain);
		if ($localLen < 1 || $localLen > 64)
		{
			// local part length exceeded
			$isValid = false;
		}
		else if ($domainLen < 1 || $domainLen > 255)
		{
			// domain part length exceeded
			$isValid = false;
		}
		else if ($local[0] == '.' || $local[$localLen-1] == '.')
		{
			// local part starts or ends with '.'
			$isValid = false;
		}
		else if (preg_match('/\\.\\./', $local))
		{
			// local part has two consecutive dots
			$isValid = false;
		}
		else if (!preg_match('/^[A-Za-z0-9\\-\\.]+$/', $domain))
		{
			// character not valid in domain part
			$isValid = false;
		}
		else if (preg_match('/\\.\\./', $domain))
		{
			// domain part has two consecutive dots
			$isValid = false;
		}
		else if
		(!preg_match('/^(\\\\.|[A-Za-z0-9!#%&`_=\\/$\'*+?^{}|~.-])+$/',
				str_replace("\\\\","",$local)))
		{
			// character not valid in local part unless
			// local part is quoted
			if (!preg_match('/^"(\\\\"|[^"])+"$/',
					str_replace("\\\\","",$local)))
			{
				$isValid = false;
			}
		}
		if(CHECK_LINK == 1) {
			if ($isValid && !(checkdnsrr($domain,"MX") || checkdnsrr($domain,"A")))
			{
				// domain not found in DNS
				$isValid = false;
			}
		}
	}
	return $isValid;
}

/**
 * commentEnabled($m)
 * Check if comment is enabled on
 * given module
 * @param string
 * @return boolean
 **/
function commentEnabled($m)
{
	if(!defined('MOD_COMMENT') || MOD_COMMENT == 0) {
		return false;
	} else {
		if(is_numeric(strpos(COMMENT_MODULES, $m))) return true;
		else return false;
	}
}

/**
 * setBreadcrumb()
 * Set breadcrumb content
 * @param array
 * @return void
 **/
function setBreadcrumb($array)
{
	if(!isset($GLOBALS['breadcrumb'])) $GLOBALS['breadcrumb']= array();
	$GLOBALS['breadcrumb'] = array_merge($GLOBALS['breadcrumb'], $array);
}

/**
 * getBreadcrumb()
 * get the Breadcrumb for displaying
 * @param string (optional)
 * @return string
 **/
function getBreadcrumb($sep = '»')
{
	$str = '<div id="breadcrumb"><div class="breadcrumb-container">'.PHP_EOL;
	$str.= '<ul>'.PHP_EOL;
	$str.= '<li><span class="separator">'.$sep.' </span><a href="'.SITE_ROOT_URL.'public/">'._t('way', 'home').'</a></li>'.PHP_EOL;
	foreach($GLOBALS['breadcrumb'] as $key => $value) {

		if($value!=false) $str.= '<li><span class="separator">'.$sep.' </span><a href="'.$value.'">'.$key.'</a></li>'.PHP_EOL;
		else $str.= '<li><span class="separator">'.$sep.' </span>'.$key.'</li>'.PHP_EOL;
	}
	$str.= '</ul>'.PHP_EOL;
	$str.= '</div></div>'.PHP_EOL;

	return $str;
}

/**
 * GetAllFiles()
 * Return an array of filenames
 * @param string
 * @param array
 * @param boolean
 * @return array
 **/
function GetAllFiles($folder,$ext=array('txt'),$recursif=true) {

	$files = array();
	$dir=opendir($folder);
	while ($file = readdir($dir)) {
		if ($file == '.' || $file == '..') continue;
		if (is_dir($folder.'/'.$file)) {
			if ($recursif==true)
				$files=array_merge($files, GetAllFiles($folder.'/'.$file, $ext));
		} else {
			foreach ($ext as $v) {
				if (strtolower($v)==strtolower(substr($file,-strlen($v)))) {
					$files[] = $folder.'/'.$file;
					break;
				}
			}
		}
	}
	closedir($dir);
	return $files;
}


/**
 * ListDir()
 * Return an HTML list for a given folder
 * @param int
 * @param string
 * @param string
 * @param boolean
 * @return string
 **/
function ListDir($dir_handle,$path, $url) {

	$i= 0;
	$dirFiles = array();

	// we prepare files
	while (false !== ($file = readdir($dir_handle))) {
		$dir = $path . $file;
		if(is_dir($dir) && $file != '.' && $file !='..' ) {
			$handle = @opendir($dir) or die("Unable to open file $file");
			ListDir($handle, $dir.'/', $url);
		} elseif($file != '.' && $file !='..' && $file !='.htaccess') {
			array_push($dirFiles, $dir);
		}
	}
	closedir($dir_handle);

	// sort by name
	sort($dirFiles);

	// uncomment to reverse array
	// $dirFiles= array_reverse($dirFiles);

	// uncomment to sort on latest modification
	//    array_multisort(
	//      array_map( 'filemtime', $dirFiles ),
	//      SORT_NUMERIC,
	//      SORT_DESC,
	//      $dirFiles
	//    );

	echo "<ul>";
	foreach($dirFiles as $dir)
	{
		$size= formatBytes(filesize($dir), 2);
		$class = ($i++ & 1) ? '' : 'class="odd"';
		$dir = str_replace('../', '', $dir);
		$dir = str_replace('//', '/', $dir);
		$dir = dirname($dir).'/'. rawurlencode(basename($dir));
		$file = basename(rawurldecode($dir));

		$comment_actions = '';

		if(MOD_COMMENT == 1 && commentEnabled('files')) {
			$id = str_replace('library/userfiles/', '', $dir);
			$nb = getFileComments($id);
			$comment_actions .= '<span class="comment_actions">';
			$comment_actions .= '<a class="comment_see" href="../comment/_ajax_view.php?id='.$id.'&amp;module=files&amp;order_by=ASC" title="' . $file . ' - ' . _t('comment','go_to') . ' / ' . _t('comment','add') .'"><span class="cc">' . $nb . '</span><span class="textcc"> ' . _t('comment','comments').' - ' . _t('comment','add') .'</span></a>';
			$comment_actions .= '</span>';
		}

		echo '<li '.$class.'><a href="'. $url . $dir .'">'.$file.'</a> <em>('.$size.')</em> '.$comment_actions.'<a  class="dlfile" href="'. $url .'library/dl.php?file='. $dir .'" title="'._t('divers','dl'). ' ' . $file.'"><span>' ._t('divers','dl'). '</span></a></li>';
	}


	if($i==0) echo '<li>'._t('divers','no_files').'</li>';
	echo "</ul>";
}

/**
 * get_cron_job()
 * @param string $command
 * @param string $part (optionnal)
 * @return array
 */
function get_cron_job($command, $part = false) {

	$a = array();
	$command = trim($command);

	// we find the last occurence of space to split the string
	// and produce crontab date / crontab script
	$pos = strrpos($command, " ");
	if($pos === false) die('Invalid cronjob!');

	$a['date'] = substr ($command , 0, $pos);
	$a['script'] = substr ($command , $pos+1, strlen($command));
	
	if($part == false )	return $a;
	else return $a[$part];
}

/**
 * formatBytes()
 * Make File size readable
 * @param int
 * @param int (optional)
 * @return string
 **/
function formatBytes($bytes, $precision = 2) {
	$units = array('o', 'Ko', 'Mo', 'Go', 'To');

	$bytes = max($bytes, 0);
	$pow = floor(($bytes ? log($bytes) : 0) / log(1024));
	$pow = min($pow, count($units) - 1);

	$bytes /= pow(1024, $pow);

	return round($bytes, $precision) . ' ' . $units[$pow];
}

/**
 * logfile()
 * Log into file
 * @param string
 * @param array
 * @return void
 **/
function logfile($src, $a) {

	$sep = '##';
	$fp = @fopen($src, 'a');

	@fwrite($fp, date('[d-m-y H:i:s]') . $sep);

	foreach($a as $value) {
		@fwrite($fp, $value . $sep);
	}

	@fwrite($fp, PHP_EOL);
	@fclose($fp);

	return true;
}

/**
 * getThemes()
 * Return installed themes
 * into an array
 * @param string
 * @return array
 **/
function getThemes($f) {

	$a = array();

	if ($handle = opendir('../templates/'.$f)) {
		$sep='';
		while (false !== ($file = readdir($handle))) {
			if ($file != "." && $file != ".." && $file != ".svn") {
				array_push($a, $file);
			}
		}
		closedir($handle);
	}

	return $a;
}

/*
 * @param $str : html string
 * @return $html html string
 */
function clean_html_body( $str ) {
	
	// See HTML safe tags for emails
	// @url : http://www.pinpointe.com/blog/email-campaign-html-and-css-support
	
	// Create DOM from string using SIMPLEHTMLDOM
	$html = str_get_html($str);
	
	$tags = array('iframe', 'object', 'embed', 'video', 'audio');
	
	foreach($tags as $tag) {
		foreach($html->find($tag) as $element) {
			$images = $element->outertext = _t('mail', 'online_content_only');
		}
	}
	
	return $html;
	
}

/**
 * notifyUsersMsg()
 * @access public
 * @param array $a
 * $a['owner'] : item owner
 * $a['user'] : the one who posted the item (can be an adminsitrator or animator)
 * @return boolean
 */
function notifyUsersMsg($a) {
	
		include_once(override('../mail/mail_actions.php'));
		include_once(override('../mail/send.php'));
		
		include_once('../class/class.emailq.php');
		$emailqo = new emailq;

		if($a['action'] == 'workgroup_request') {
			$m = array('O');
		
			$wg = $a['id'];
			$data = $GLOBALS['sql_object']->DBSelect(SQL_getWorkshopDenomination($a['id']));
			if(count($data)==1) $a['wgroups'] = $data[0]['workshop_denomination'];
			$a['comment'] = strip_tags($_POST ['comment']);
			
			$users = $GLOBALS['sql_object'] -> DBSelect(SQL_getWorkshopUserList($a['id'], array('O')), 'OBJECT');

		}
		
    if($a['action'] == 'post_alert') {
      if(ALERT_NEWPOST == 0) return true;
      if(ALERT_NEWPOST == 1) $m = array('O', 'A');
      if(ALERT_NEWPOST == 2) $m = array('O', 'A', 'U');

      $a['topic_body'] = clean_html_body($a['topic_body']);
      
      $wg = $a['parentid'];
      $data = $GLOBALS['sql_object']->DBSelect(SQL_getWorkshopDenomination($a['parentid']));
      if(count($data)==1) $a['wname'] = $data[0]['workshop_denomination'];
      $a['wlinkforum'] = array('rub'=> $GLOBALS['links'][U_L]['topic']['linkvalue'],'id'=> $a['tid'], 'parentid' => $a['parentid'], 'name' => $a['wname'], '#' => 'msg-'.$a['id']);
      $users = $GLOBALS['sql_object'] -> DBSelect(SQL_getWorkshopUsersforNotification($a['parentid'], 'post'), 'OBJECT');
    }

    if($a['action'] == 'topic_alert') {
      if(ALERT_NEWTOPIC == 0) return true;
      if(ALERT_NEWTOPIC == 1) $m = array('O', 'A');
      if(ALERT_NEWTOPIC == 2) $m = array('O', 'A', 'U');

      $a['topic_body'] = clean_html_body($a['topic_body']);
      
      $data = $GLOBALS['sql_object']->DBSelect(SQL_getWorkshopDenomination($a['id']));
      if(count($data)==1) $a['wname'] =$data[0]['workshop_denomination'];

      $a['wlinkforum'] = array('rub'=> $GLOBALS['links'][U_L]['topic-list']['linkvalue'],'id'=> $a['id'], 'name' => $a['wname']);

      $users = $GLOBALS['sql_object'] -> DBSelect(SQL_getWorkshopUsersforNotification($a['id'], 'topic'), 'OBJECT');
    }

    if($a['action'] == 'file_alert') {
      if(ALERT_NEWFILE == 0) return true;
      if(ALERT_NEWFILE == 1) $m = array('O', 'A');
      if(ALERT_NEWFILE == 2) $m = array('O', 'A', 'U');

      $data = $GLOBALS['sql_object']->DBSelect(SQL_getWorkshopDenomination($a['id']));
      if(count($data)==1) $a['wname'] =$data[0]['workshop_denomination'];

      $a['wlinkfiles'] = array('rub'=> $GLOBALS['links'][U_L]['files']['linkvalue'],'id'=> $a['id'], 'name' => $a['wname']);
      $a['wlinkworkgroups'] = array('rub'=> $GLOBALS['links'][U_L]['workgroup']['linkvalue']);

      if(isset($a['is_shared']) && $a['is_shared'] === true) {
      	// we notify all users
      	$users = $GLOBALS['sql_object']->DBSelect(SQL_getAllWorkshopUsersforNotification('file'), 'OBJECT');
      } else {
      	// we notify only current workgroup
      	$users = $GLOBALS['sql_object']->DBSelect(SQL_getWorkshopUsersforNotification($a['id'], 'file'), 'OBJECT');
      }
    }

    if($a['action'] == 'event_alert') {
    	if(ALERT_NEWEVENT == 0) return true;
    	if(ALERT_NEWEVENT == 1) $m = array('O', 'A');
    	if(ALERT_NEWEVENT == 2) $m = array('O', 'A', 'U');
    
    	$a['task_body'] = clean_html_body($a['task_body']);
    	
    	$data = $GLOBALS['sql_object']->DBSelect(SQL_getWorkshopDenomination($a['id']));
    	if(count($data)==1) $a['wname'] = $data[0]['workshop_denomination'];
    	
    	$a['wlinkcalendar'] = array('rub'=> $GLOBALS['links'][U_L]['calendar']['linkvalue'],'id'=> $a['id'], 'name' => $a['wname']);
    	$users = $GLOBALS['sql_object']->DBSelect(SQL_getWorkshopUsersforNotification($a['id'], 'file'), 'OBJECT');
    }

    foreach($users as $user) {
      if($user->user_login != $a['user']) {
        if(in_array($user->jwu_user_right, $m)) {
        	
        	$a['recipient'] = $user->profile_email;
        	
          $emailContainer = prepare_email($a);
          
          // if no email queue, we send mail directly
          if(!defined('MAIL_BULK') || MAIL_BULK == 'nodelay') {
          	
          	$r = send_email($emailContainer);
          
          // we prepare the email queue and store it in database
          // emails are sent later using cron
          } else {
          	 
          	$queue = array();
          	$queue = $emailContainer;
          	$queue['module'] 		= 'workgroups';
          	
          	if($a['action'] == 'event_alert') $queue['priority'] = 1;
          	 
          	$r = $emailqo->addToQueue($queue, $GLOBALS['sql_object']);
          	
          }
          
        }
      }
    }
    
    return true;
  }

/**
 * cleanString()
 * Remove exotic characters form string
 * same as used in filemanager
 * @param string
 * @param array
 * @return string
 **/
function cleanString($string, $allowed = array()) {
	$allow = null;

	if (!empty($allowed)) {
		foreach ($allowed as $value) {
			$allow .= "\\$value";
		}
	}

	$mapping = array(
			'Š'=>'S', 'š'=>'s', 'Đ'=>'Dj', 'đ'=>'dj', 'Ž'=>'Z', 'ž'=>'z', 'Č'=>'C', 'č'=>'c', 'Ć'=>'C', 'ć'=>'c',
			'À'=>'A', 'Á'=>'A', 'Â'=>'A', 'Ã'=>'A', 'Ä'=>'A', 'Å'=>'A', 'Æ'=>'A', 'Ç'=>'C', 'È'=>'E', 'É'=>'E',
			'Ê'=>'E', 'Ë'=>'E', 'Ì'=>'I', 'Í'=>'I', 'Î'=>'I', 'Ï'=>'I', 'Ñ'=>'N', 'Ò'=>'O', 'Ó'=>'O', 'Ô'=>'O',
			'Õ'=>'O', 'Ö'=>'O', 'Ő'=>'O', 'Ø'=>'O', 'Ù'=>'U', 'Ú'=>'U', 'Û'=>'U', 'Ü'=>'U', 'Ű'=>'U', 'Ý'=>'Y',
			'Þ'=>'B', 'ß'=>'Ss','à'=>'a', 'á'=>'a', 'â'=>'a', 'ã'=>'a', 'ä'=>'a', 'å'=>'a', 'æ'=>'a', 'ç'=>'c',
			'è'=>'e', 'é'=>'e', 'ê'=>'e', 'ë'=>'e', 'ì'=>'i', 'í'=>'i', 'î'=>'i', 'ï'=>'i', 'ð'=>'o', 'ñ'=>'n',
			'ò'=>'o', 'ó'=>'o', 'ô'=>'o', 'õ'=>'o', 'ö'=>'o', 'ő'=>'o', 'ø'=>'o', 'ù'=>'u', 'ú'=>'u', 'ű'=>'u',
			'û'=>'u', 'ý'=>'y', 'ý'=>'y', 'þ'=>'b', 'ÿ'=>'y', 'Ŕ'=>'R', 'ŕ'=>'r', ' '=>'_', "'"=>'_', '/'=>''
	);

	if (is_array($string)) {

		$cleaned = array();

		foreach ($string as $key => $clean) {
			$clean = strtr($clean, $mapping);
			$clean = preg_replace("/[^{$allow}_a-zA-Z0-9]/", '', $clean);
			$cleaned[$key] = preg_replace('/[_]+/', '_', $clean); // remove double underscore
		}
	} else {
		$string = strtr($string, $mapping);
		$string = preg_replace("/[^{$allow}_a-zA-Z0-9]/", '', $string);
		$cleaned = preg_replace('/[_]+/', '_', $string); // remove double underscore
	}
	return $cleaned;
}

/**
 * get_referer()
 * @return string
 */
function get_referer() {
	
	$referer = isset($_SERVER['HTTP_REFERER']) ? $_SERVER['HTTP_REFERER'] : '';
	return $referer;
	
}

// Returns the real IP address of the user
function i2c_realip()
{
	// No IP found (will be overwritten by for
	// if any IP is found behind a firewall)
	$ip = false;
	// If HTTP_CLIENT_IP is set, then give it priority
	if (!empty($_SERVER["HTTP_CLIENT_IP"])) {
		$ip = $_SERVER["HTTP_CLIENT_IP"];
	}
	// User is behind a proxy and check that we discard RFC1918 IP addresses
	// if they are behind a proxy then only figure out which IP belongs to the
	// user.  Might not need any more hackin if there is a squid reverse proxy
	// infront of apache.
	if (!empty($_SERVER['HTTP_X_FORWARDED_FOR'])) {
		// Put the IP's into an array which we shall work with shortly.
		$ips = explode (", ", $_SERVER['HTTP_X_FORWARDED_FOR']);
		if ($ip) {
			array_unshift($ips, $ip);
			$ip = false;
		}

		for ($i = 0; $i < count($ips); $i++) {
			// Skip RFC 1918 IP's 10.0.0.0/8, 172.16.0.0/12 and
			// 192.168.0.0/16 -- jim kill me later with my regexp pattern
			// below.
			if (!eregi ("^(10|172\.16|192\.168)\.", $ips[$i])) {
				$ip = $ips[$i];
				break;
			}
		}
	}
	// Return with the found IP or the remote address
	return ($ip ? $ip : $_SERVER['REMOTE_ADDR']);
}

/**
 * securityCheck()
 * Test and sanitize user input
 * from request
 * @return boolean
 **/
function securityCheck() {

	$passed = true;

	// we first sanitize vars
	if(isset($_REQUEST['rub']))
		$_REQUEST['rub']= strip_tags($_REQUEST['rub']);
	if(isset($_REQUEST['name']))
		$_REQUEST['name']= strip_tags($_REQUEST['name']);
	if(isset($_REQUEST['search']))
		$_REQUEST['search']= strip_tags($_REQUEST['search']);
	if(isset($_REQUEST['newsletteremail']))
		$_REQUEST['newsletteremail']= strip_tags($_REQUEST['newsletteremail']);
	// then do tests
	if(isset($_REQUEST['id']) && preg_match('/[^0-9A-Za-z]/',$_REQUEST['id']))
		$passed= false;
	if(isset($_REQUEST['parentid']) && !is_numeric($_REQUEST['parentid']))
		$passed= false;
	if(isset($_REQUEST['parentparentid']) && !is_numeric($_REQUEST['parentparentid']))
		$passed= false;
	if(isset($_REQUEST['debut']) && !is_numeric($_REQUEST['debut']))
		$passed= false;

	if($passed == false) die('no way!');
	else return true;
}
?>